param(
    [Parameter(Mandatory=$true)]
    [string]$SourcePath,
    
    [Parameter(Mandatory=$true)]
    [string]$DestinationPath,
    [string[]]$ExcludeFolders = @('node_modules', '.next', 'dist', 'build', '.git'),
    [string[]]$ExcludeFiles = @('package-lock.json'),
    [string[]]$IncludeExtensions = @('.tsx', '.ts', '.js', '.jsx', '.css', '.scss', '.json')
)

# Create destination directory if it doesn't exist
if (-not (Test-Path -Path $DestinationPath)) {
    New-Item -ItemType Directory -Path $DestinationPath | Out-Null
}

# Keep track of filenames and their full paths
$fileTracker = @{}

# First pass: collect all files and their paths
function CollectFiles {
    param (
        [string]$path
    )
    
    try {
        # Use -LiteralPath to handle special characters
        Get-ChildItem -LiteralPath $path -Force | ForEach-Object {
            $currentFile = $_
            
            # Handle directories
            if ($currentFile.PSIsContainer) {
                if (-not ($ExcludeFolders -contains $currentFile.Name)) {
                    # Recursively process subfolders using literal path
                    CollectFiles -path $currentFile.FullName
                }
            }
            # Handle files
            else {
                if (($IncludeExtensions -contains $currentFile.Extension) -and 
                    (-not ($ExcludeFiles -contains $currentFile.Name))) {
                    
                    $fileName = $currentFile.Name
                    if (-not $fileTracker.ContainsKey($fileName)) {
                        $fileTracker[$fileName] = @()
                    }
                    $fileTracker[$fileName] += $currentFile.FullName
                }
            }
        }
    }
    catch {
        Write-Host "Error processing path: $path"
        Write-Host $_.Exception.Message
    }
}

# Function to get meaningful path segments
function Get-PathSegments {
    param (
        [string]$fullPath
    )
    
    $relativePath = $fullPath.Replace($SourcePath, '').TrimStart([IO.Path]::DirectorySeparatorChar)
    $segments = $relativePath.Split([IO.Path]::DirectorySeparatorChar)
    return $segments | Where-Object { $_ -and ($ExcludeFolders -notcontains $_) }
}

# Function to generate a unique name based on the file path
function Get-UniqueFileName {
    param (
        [string]$fullPath,
        [string]$originalFileName,
        [int]$index,
        [int]$total
    )
    
    if ($total -eq 1) {
        return $originalFileName
    }
    
    $extension = [System.IO.Path]::GetExtension($originalFileName)
    $nameWithoutExt = [System.IO.Path]::GetFileNameWithoutExtension($originalFileName)
    
    # Get path segments
    $segments = Get-PathSegments -fullPath $fullPath
    
    $suffix = ""
    # Try to create a meaningful name based on path depth
    if ($segments.Count -gt 0) {
        # Start from the end and work backwards to find a meaningful folder name
        for ($i = $segments.Count - 1; $i -ge 0; $i--) {
            $currentSegment = $segments[$i]
            # Skip the current filename and any bracketed segments
            if ($currentSegment -ne $originalFileName -and -not $currentSegment.StartsWith('[')) {
                $suffix = $currentSegment.ToLower()
                break
            }
        }
        
        # If we couldn't find a meaningful segment, use the index
        if ([string]::IsNullOrEmpty($suffix)) {
            $suffix = $index.ToString()
        }
    } else {
        # If we don't have any path segments, just use a number
        $suffix = $index.ToString()
    }
    
    return "{0}_{1}{2}" -f $nameWithoutExt, $suffix, $extension
}

# Collect all files first
Write-Host "Collecting files..."
CollectFiles -path $SourcePath

# Process and copy files with proper naming
Write-Host "Processing files..."
foreach ($fileName in $fileTracker.Keys) {
    $paths = $fileTracker[$fileName]
    $total = $paths.Count
    
    for ($i = 0; $i -lt $total; $i++) {
        $sourcePath = $paths[$i]
        try {
            $newFileName = Get-UniqueFileName -fullPath $sourcePath -originalFileName $fileName -index ($i + 1) -total $total
            $destinationFile = Join-Path -Path $DestinationPath -ChildPath $newFileName
            
            # Copy the file using -LiteralPath
            Copy-Item -LiteralPath $sourcePath -Destination $destinationFile -Force
            Write-Host "Copied: $sourcePath -> $newFileName"
        }
        catch {
            Write-Host "Error processing file: $sourcePath"
            Write-Host $_.Exception.Message
        }
    }
}

Write-Host "`nDone! Files have been flattened to: $DestinationPath"